//###########################################################################
//
// FILE:  Example_Flash281x_API.c	
//
// TITLE: F281x Flash API Example
//
// NOTE:  This example runs from Flash.  First program the example
//        into flash.  The code will then copy the API's to RAM and 
//        modify the flash. 
//
//
//###########################################################################
// $TI Release: Flash281x API V2.10 $
// $Release Date: August 4, 2005 $
//###########################################################################


/*---- Flash API include file -------------------------------------------------*/
#include "Flash281x_API_Library.h"

/*---- example include file -------------------------------------------------*/
#include "Example_Flash281x_API.h"
#include "SCI_Boot.h"

extern Uint16 SCIA_GetByteData_app(void);
extern Uint16 SCIA_FIFO_GetWordData_swapEndian(void);
void SCIA_BlockProcessing(void);
extern void SCI_SendStatus(char *);
extern void SCIA_AutobaudLock(void);
extern void SCIA_Init(void);
int checksum_flash(void);
void initPLL(void);
void main2(void);
extern void	SCI_SendHexLong(Uint32);
extern	long Convert_HEXtoASCII(Uint16);



/*--- Global variables used to interface to the flash routines */
#define	CHECKSUM_EXPECTED 0x6E13

#if		FLASH_F2810
#define FLASH_RANGE       65535L
#define DEST_ADDR		  0x003E8000
#endif

#if		FLASH_F2811 | FLASH_F2812
#define FLASH_RANGE       131071L
#define DEST_ADDR		  0x003D8000
#endif

#define BLOCK_SIZE			FLASH_RANGE+1L
#define RAM_BUFFER_SIZE		4096


FLASH_ST EraseStatus;
FLASH_ST ProgStatus;
FLASH_ST VerifyStatus; 

#pragma DATA_SECTION(BlockBuffer1, "BlockTransferBuffer1");
#pragma DATA_SECTION(BlockBuffer2, "BlockTransferBuffer2");
Uint16 	BlockBuffer1[4096];
Uint16 	BlockBuffer2[4096];
Uint16	*BlockBufferPtr;
Uint32	BlockRemain;
Uint16	BufferIndexMax;
Uint32 MyCallbackCounter; // Just increment a counter in the callback function
Uint32	tempL;
Uint16 	i=0;
                               
/*--- Callback function.  Function specified by defining Flash_CallbackPtr */
void MyCallbackFunction(void); 
  
void main2(void)
{   
   	Uint16 ErrorFlag, temp, Status;

/*------------------------------------------------------------------
 To use the F2812, F2811 or F2810 Flash API, the following steps
 must be followed:

      1. Modify Flash281x_API.config.h for your targets operating
         conditions.
      2. Include Flash281x_API_Library.h in the application.
      3. Add the approparite Flash API library to the project.

  The user's code is responsible for the following:

      4. Initalize the PLL to the proper CPU operating frequency.
      5. If required, copy the flash API functions into on-chip zero waitstate
         RAM.  
      6. Initalize the Flash_CPUScaleFactor variable to SCALE_FACTOR 
      7. Initalize the flash callback function pointer. 
      8. Optional: Run the Toggle test to confirm proper frequency configuration
         of the API. 
      9. Optional: Unlock the CSM.  
     10. Call the API functions: Flash_Erase(), Flash_Program(), Flash_Verify()
         
  The API functions will:
      
       Disable the watchdog
       Check the device revision (REVID).  This API is for Revision C silicon
       Perform the desired operation and return status
------------------------------------------------------------------*/

	SCI_SendStatus("Communication kernel received and executing.");
	SCI_SendStatus("\n\rType 'a' to relock baud-rate: ");


/*------------------------------------------------------------------
 Initalize the PLLCR value before calling any of the F2810, F2811
 or F281x Flash API functions.
        
     Check to see if the PLL needs to changed
     PLLCR_VALUE is defined in Example_Flash281x_API.h
     1) Make the change
     2) Wait for the DSP to switch to the PLL clock
        This wait is performed to ensure that the flash API functions 
        will be executed at the correct frequency.
     3) While waiting, feed the watchdog so it will not reset. 
------------------------------------------------------------------*/

    initPLL();

/*------------------------------------------------------------------
  Initalize Flash_CPUScaleFactor.

   Flash_CPUScaleFactor is a 32-bit global variable that the flash
   API functions use to scale software delays. This scale factor 
   must be initalized to SCALE_FACTOR by the user's code prior
   to calling any of the Flash API functions. This initalization
   is VITAL to the proper operation of the flash API functions.  
   
   SCALE_FACTOR is defined in Example_Flash281x_API.h as   
     #define SCALE_FACTOR  1048576.0L*( (200L/CPU_RATE) )
     
   This value is calculated during the compile based on the CPU 
   rate, in nanoseconds, at which the algorithums will be run.
------------------------------------------------------------------*/
   
   Flash_CPUScaleFactor = SCALE_FACTOR;


/*------------------------------------------------------------------
  Initalize Flash_CallbackPtr.

   Flash_CallbackPtr is a pointer to a function.  The API uses
   this pointer to invoke a callback function during the API operations.
   If this function is not going to be used, set the pointer to NULL
   NULL is defined in <stdio.h>.  
------------------------------------------------------------------*/
   
   Flash_CallbackPtr = &MyCallbackFunction; 
   MyCallbackCounter = 0; // Increment this counter in the callback function

	//Prep SCI with new baud rate
	SCIA_Init();
	SCIARegs.SCIFFTX.bit.SCIFFENA = 0;
	SCIA_AutobaudLock();
	SCIARegs.SCICTL1.bit.SWRESET = 0;	
	SCIARegs.SCICTL1.bit.SWRESET = 1;
	SCIARegs.SCIFFTX.bit.SCIFFENA = 1;
	SCIARegs.SCIFFTX.all &= 0x7FFF;
	SCIARegs.SCIFFTX.all |= 0x8000;
	SCI_SendStatus(" baud-rate relocked\n\r\r ");

	//Erase flash
	temp = checksum_flash();			
	tempL = Convert_HEXtoASCII(temp);
	SCI_SendStatus("\n\n\rFlash Checksum = 0x");
	SCI_SendHexLong(tempL);
	if(temp==0)
		SCI_SendStatus("\n\rFLASH appears to be erased already.");
	else
		SCI_SendStatus("\n\rFLASH is not erased and should not be programmed until it is erased.");
	SCI_SendStatus("\n\n\rErase FLASH? (y/n): ");
	temp = SCIA_GetByteData_app(); 
    if( temp != 'n')
	{
		SCI_SendStatus("\n\rErasing ... please wait");
		#if FLASH_F2810
				Status = Flash_Erase(SECTOR_F2810,&EraseStatus);
		#endif

		#if FLASH_F2811 | FLASH_F2812
				Status = Flash_Erase(SECTOR_F2811,&EraseStatus);
		#endif
		if(Status != STATUS_SUCCESS) 
		{
		   SCI_SendStatus("\n\r Flash Erase Error ");
		   Example_Error(Status);
		}
		SCI_SendStatus(" ** erasing done\n\r");
	}
  
	//Transfer and program application code
	SCI_SendStatus("\n\rReady for application data transfer...\n\r");    
   	SCIA_BlockProcessing();

	//Calculate new checksum and compare with expected value
	SCI_SendStatus(" ** application programmed");
	temp = checksum_flash();
	tempL = Convert_HEXtoASCII(temp);
	SCI_SendStatus("\n\rFlash Checksum = 0x");
	SCI_SendHexLong(tempL);
	ErrorFlag = temp - CHECKSUM_EXPECTED;
	if(ErrorFlag)
   	{
		SCI_SendStatus("  ** checksum error ");
   		Example_Error(100);
	}
	SCI_SendStatus("  ** checksum verified ");

	//CKFA software done
	Example_Done();
}

int checksum_flash()
{
    unsigned long   j;
    unsigned int    sum = 0;
	unsigned int	*mem;

	mem = (unsigned int*)DEST_ADDR;
    for (j = 0; j <= FLASH_RANGE; j++)
		sum += *mem++;

	return sum;
}


void SCIA_BlockProcessing(void)
{
   struct HEADER {
     Uint32 BlockSize;
     Uint32 DestAddr;
   } BlockHeader;
  
	Uint16	BufferSize;
	Uint16 Status;
	Uint16 *Flash_ptr;     // Pointer to a location in flash
	Uint32 Length;         // Number of 16-bit values to be programmed

	//HW definitions, RAM buffer is constant, block size and dest
	//addr dependent on 2810 or 2811
	BufferSize = RAM_BUFFER_SIZE;
	BlockHeader.BlockSize = BLOCK_SIZE;
	BlockHeader.DestAddr = DEST_ADDR;

	//Init processing variables
	BlockRemain = BlockHeader.BlockSize;
	BufferIndexMax = BufferSize;
	BlockBufferPtr = BlockBuffer1;

	//Transfer anf program application code
	while(BufferIndexMax > 0)
	{
		SCI_SendStatus(".");
		for(i = 0; i <= (BufferIndexMax-1); i++)
		{
			//RAM Transfer
			*BlockBufferPtr++ = SCIA_FIFO_GetWordData_swapEndian();
			if(SCIARegs.SCIRXST.bit.RXERR || SCIARegs.SCIFFRX.bit.RXFFOVF)
			{
			    SCI_SendStatus("\n\rSerial-Comm Failure - 1");
		   		Example_Error(101);
			}
		}			                          
		
		//Program data block into flash at address specified at start of data block,
		//callback function continues transfer in other buffer

		//Config RAM buffer pointer for next RAM transfer
		if(BlockBufferPtr <= BlockBuffer2)
			BufferIndexMax = BlockBufferPtr - BlockBuffer1;
		else
			BufferIndexMax = BlockBufferPtr - BlockBuffer2;

		Flash_ptr = (Uint16 *)BlockHeader.DestAddr;
		Length = BufferIndexMax;
		BlockHeader.DestAddr += Length;

		//Callback function config
		MyCallbackCounter = 0; 
		BlockRemain -= BufferIndexMax;

		//Program flash
		if(BlockRemain < BufferSize) 
			BufferIndexMax = BlockRemain;
		else
			BufferIndexMax = BufferSize;

		if(BlockBufferPtr <= BlockBuffer2)
		{
			BlockBufferPtr = BlockBuffer2;
			Status = Flash_Program(Flash_ptr,BlockBuffer1,Length,&ProgStatus);
		}
		else
		{
			BlockBufferPtr = BlockBuffer1;
			Status = Flash_Program(Flash_ptr,BlockBuffer2,Length,&ProgStatus);
		}
		if(Status != STATUS_SUCCESS) 
		{
			SCI_SendStatus("\n\rFlash programming failure");
		   	Example_Error(Status);

		}

		//Receive the rest of data block not transferred by callback function
		// and store in buffer 2
		while((SCIARegs.SCIFFRX.bit.RXFIFST>1) && (BufferIndexMax>0))
		{
			*BlockBufferPtr++ = SCIA_FIFO_GetWordData_swapEndian();
			BufferIndexMax--;
		}
		if(SCIARegs.SCIRXST.bit.RXERR || SCIARegs.SCIFFRX.bit.RXFFOVF)
		{
			if(BlockRemain == 0)
			{
				SCI_SendStatus("\n\rChange configuration to 128 kW FLASH ");
				Example_Error(102);
			}
			else
			{
				SCI_SendStatus("\n\rSerial-Comm Failure - 2 ");
				Example_Error(103);
		   	}		   	
		}

	}	//end of IndexMax While-Loop

   return;
}


/*------------------------------------------------------------------
  Callback function - must be executed from outside flash/OTP
-----------------------------------------------------------------*/
//#pragma CODE_SECTION(MyCallbackFunction,"ramfuncs");
void MyCallbackFunction(void)
{       

	if(SCIARegs.SCIFFRX.bit.RXFIFST == 16)
	{
		while((SCIARegs.SCIFFRX.bit.RXFIFST>1) && (BufferIndexMax>0))
		{
			*BlockBufferPtr++ = SCIA_FIFO_GetWordData_swapEndian();
			BufferIndexMax--;
		}
		if(SCIARegs.SCIRXST.bit.RXERR || SCIARegs.SCIFFRX.bit.RXFFOVF)
		{
			if(BlockRemain == 0)
			{
				SCI_SendStatus("\n\rChange configuration to 128 kW FLASH ");
				Example_Error(104);
			}
			else
			{
				SCI_SendStatus("\n\rSerial-Comm Failure - 2 ");
				Example_Error(105);
		   	}		   	
		}

	}
    // Toggle pin, service external watchdog etc
    MyCallbackCounter++;
}


/*------------------------------------------------------------------
  Simple memory copy routine to move code out of flash into SARAM
-----------------------------------------------------------------*/

void Example_MemCopy(Uint16 *SourceAddr, Uint16* SourceEndAddr, Uint16* DestAddr)
{
    while(SourceAddr < SourceEndAddr)
    { 
       *DestAddr++ = *SourceAddr++;
    }
    return;
}


/*------------------------------------------------------------------
  For this example, if an error is found just stop here
-----------------------------------------------------------------*/
void Example_Error(Uint16 Status)
{

//  Error code will be in the AL register. 
    asm("    ESTOP0");
    asm("    SB 0, UNC");
}


/*------------------------------------------------------------------
  For this example, once we are done just stop here
-----------------------------------------------------------------*/
//#pragma CODE_SECTION(Example_Done,"ramfuncs");
void Example_Done(void)
{
    asm("    ESTOP0");
    asm("    SB 0, UNC");
}


void initPLL(void)
{
    int	i;
    if(*PLLCR != PLLCR_VALUE) 
    {
       EALLOW;
       *PLLCR = PLLCR_VALUE;
       
       // Wait for PLL to lock
       // Each time through this loop takes ~14 cycles
       // PLL Lock time is 131072 Cycles
       for(i= 0; i< 131072/14; i++){
           *WDKEY = 0x0055;
           *WDKEY = 0x00AA;
       }
       for(i= 0; i< 131072/14; i++){
           *WDKEY = 0x0055;
           *WDKEY = 0x00AA;
       }
       EDIS;
    }

	return;
}

/*------------------------------------------------------------------
   Example_ToggleTest
  
   This function shows example calls to the ToggleTest.  

   This test is used to Toggle a GPIO pin at a known rate and thus 
   confirm the frequency configuration of the API functions.
   
   Common output pins are supported here, however the user could
   call the function with any GPIO mux register and pin mask.
   
   A pin should be selected based on the hardware being used. 
   
   Parameters: Uint16 PinNumber

   Return Value: The toggle test does not return.  It will loop 
                 forever and is used only for testing purposes.
        
   Notes:
----------------------------------------------------------------*/
/*
void Example_ToggleTest(Uint16 PinNumber)
{

       switch(PinNumber) {
          case 0:
             break;
          case 1:
             Flash_ToggleTest(GPFMUX,GPFTOGGLE,GPIOF14_XF_MASK);
             break;
          case 2:
             Flash_ToggleTest(GPAMUX,GPATOGGLE,GPIOA0_PWM1_MASK);
             break;
          case 3:
             Flash_ToggleTest(GPFMUX,GPFTOGGLE,GPIOF4_SCITXDA_MASK);
             break;
          case 4:
             Flash_ToggleTest(GPGMUX,GPGTOGGLE,GPIOG4_SCITXDB_MASK);
             break;
          case 5:
             Flash_ToggleTest(GPFMUX,GPFTOGGLE,GPIOF12_MDXA_MASK);
             break;
          default:
             break;
       }          
}


*/


